import Foundation
import HealthKit
import WatchKit

class HealthKitService: ObservableObject {
    private let healthStore = HKHealthStore()
    
    // MARK: - Published Properties
    @Published var currentHealthData: HealthDataSnapshot?
    @Published var isAuthorized: Bool = false
    @Published var isLoading: Bool = false
    @Published var lastError: HealthKitError?
    @Published var lastUpdateTime: Date?
    
    // Health data types we want to read
    private let readTypes: Set<HKObjectType> = [
        // Heart rate metrics
        HKQuantityType.quantityType(forIdentifier: .restingHeartRate)!,
        HKQuantityType.quantityType(forIdentifier: .heartRateVariabilitySDNN)!,
        
        // Fitness metrics
        HKQuantityType.quantityType(forIdentifier: .vo2Max)!,
        HKQuantityType.quantityType(forIdentifier: .stepCount)!,
        
        // Body composition
        HKQuantityType.quantityType(forIdentifier: .bodyMass)!,
        HKQuantityType.quantityType(forIdentifier: .bodyMassIndex)!,
        
        // Sleep data
        HKCategoryType.categoryType(forIdentifier: .sleepAnalysis)!,
        
        // Demographic data
        HKCharacteristicType.characteristicType(forIdentifier: .biologicalSex)!,
        HKCharacteristicType.characteristicType(forIdentifier: .dateOfBirth)!
    ]
    
    // Health data types we want to write (for storing our analysis results)
    private let writeTypes: Set<HKSampleType> = []
    
    var isHealthKitAvailable: Bool {
        return HKHealthStore.isHealthDataAvailable()
    }
    
    func requestAuthorization() async throws {
        guard isHealthKitAvailable else {
            throw HealthKitError.notAvailable
        }
        
        do {
            // Request authorization and wait for completion
            try await healthStore.requestAuthorization(toShare: writeTypes, read: readTypes)
            
            // Add a small delay to ensure authorization status is updated
            try await Task.sleep(nanoseconds: 500_000_000) // 0.5 seconds
            
            // Verify that we have authorization for critical data types
            await verifyAuthorization()
            
        } catch {
            print("HealthKit authorization error: \(error)")
            // Don't immediately throw - let the calling method check the final status
            // This allows the app to handle previously denied permissions more gracefully
        }
    }
    
    /// Request only essential permissions for basic functionality
    func requestEssentialPermissions() async throws {
        guard isHealthKitAvailable else {
            throw HealthKitError.notAvailable
        }
        
        let essentialReadTypes: Set<HKObjectType> = [
            HKQuantityType.quantityType(forIdentifier: .restingHeartRate)!,
            HKQuantityType.quantityType(forIdentifier: .stepCount)!,
            HKCharacteristicType.characteristicType(forIdentifier: .biologicalSex)!,
            HKCharacteristicType.characteristicType(forIdentifier: .dateOfBirth)!
        ]
        
        do {
            // Request authorization for essential permissions only
            try await healthStore.requestAuthorization(toShare: writeTypes, read: essentialReadTypes)
            
            // Add a small delay to ensure authorization status is updated
            try await Task.sleep(nanoseconds: 500_000_000) // 0.5 seconds
            
            // Verify that we have authorization for critical data types
            await verifyAuthorization()
            
        } catch {
            print("Essential HealthKit authorization error: \(error)")
            // Don't immediately throw - let the calling method check the final status
        }
    }
    
    private func verifyAuthorization() async {
        // Check authorization status for critical health data types
        let criticalTypes: [(HKObjectType, String)] = [
            (HKQuantityType.quantityType(forIdentifier: .restingHeartRate)!, "HKQuantityTypeIdentifierRestingHeartRate"),
            (HKQuantityType.quantityType(forIdentifier: .stepCount)!, "HKQuantityTypeIdentifierStepCount"),
            (HKQuantityType.quantityType(forIdentifier: .bodyMass)!, "HKQuantityTypeIdentifierBodyMass"),
            (HKCharacteristicType.characteristicType(forIdentifier: .biologicalSex)!, "HKCharacteristicTypeIdentifierBiologicalSex"),
            (HKCharacteristicType.characteristicType(forIdentifier: .dateOfBirth)!, "HKCharacteristicTypeIdentifierDateOfBirth")
        ]
        
        for (type, identifier) in criticalTypes {
            let status = healthStore.authorizationStatus(for: type)
            let statusDescription = getAuthorizationStatusDescription(status)
            print("Authorization status for \(identifier): \(status.rawValue) (\(statusDescription))")
        }
    }
    
    private func getAuthorizationStatusDescription(_ status: HKAuthorizationStatus) -> String {
        switch status {
        case .notDetermined:
            return "notDetermined"
        case .sharingDenied:
            return "sharingDenied"
        case .sharingAuthorized:
            return "sharingAuthorized"
        @unknown default:
            return "unknown"
        }
    }
    
    func getAuthorizationStatus(for type: HKObjectType) -> HKAuthorizationStatus {
        return healthStore.authorizationStatus(for: type)
    }
    
    func hasRequiredPermissions() -> Bool {
        // Check if we have at least the essential permissions
        let essentialTypes: [(HKObjectType, String)] = [
            (HKQuantityType.quantityType(forIdentifier: .restingHeartRate)!, "Resting Heart Rate"),
            (HKQuantityType.quantityType(forIdentifier: .stepCount)!, "Step Count"),
            (HKCharacteristicType.characteristicType(forIdentifier: .biologicalSex)!, "Biological Sex"),
            (HKCharacteristicType.characteristicType(forIdentifier: .dateOfBirth)!, "Date of Birth")
        ]
        
        var hasAllPermissions = true
        
        for (type, name) in essentialTypes {
            let status = healthStore.authorizationStatus(for: type)
            let isAuthorized = status == .sharingAuthorized
            print("Permission check for \(name): \(getAuthorizationStatusDescription(status)) - \(isAuthorized ? "GRANTED" : "DENIED")")
            
            if !isAuthorized {
                hasAllPermissions = false
            }
        }
        
        print("Overall permission status: \(hasAllPermissions ? "GRANTED" : "DENIED")")
        return hasAllPermissions
    }
    
    /// Initialize HealthKit service and request authorization
    @MainActor
    func initializeHealthKit() async throws -> HealthDataSnapshot? {
        guard isHealthKitAvailable else {
            throw HealthKitError.notAvailable
        }
        
        print("Starting HealthKit initialization...")
        
        // Check if we can use cached test data to avoid re-running authorization
        // Only use cached data if permissions are actually granted
        if canUseCachedTestData() && hasRequiredPermissions() {
            print("Using cached health data - permissions are granted")
            return currentHealthData
        }
        
        do {
            // First, check current authorization status
            print("Checking current authorization status...")
            printPermissionStatus()
            await verifyAuthorization()
            
            // Check if essential permissions were previously denied
            let deniedEssentialPermissions = getDeniedPermissions()
            if !deniedEssentialPermissions.isEmpty {
                print("Essential permissions were previously denied: \(deniedEssentialPermissions)")
                print("Cannot request permissions again - user must manually enable in Health app")
                throw HealthKitError.permissionsDenied(deniedEssentialPermissions)
            }
            
            // Request authorization (only if not previously denied)
            print("Requesting HealthKit authorization...")
            try await requestAuthorization()
            
            // Check authorization status again after request
            print("Checking authorization status after request...")
            printPermissionStatus()
            await verifyAuthorization()
            
            // Update authorization status
            isAuthorized = hasRequiredPermissions()
            print("HealthKit authorization status: \(isAuthorized)")
            
            if isAuthorized {
                print("HealthKit authorization successful - fetching initial data...")
                // Fetch initial health data
                let healthData = try await fetchLatestHealthData()
                print("HealthKit authorization and initial data fetch completed successfully")
                return healthData
            } else {
                let missingPermissions = getMissingPermissions()
                print("HealthKit authorization incomplete - missing permissions: \(missingPermissions)")
                
                // Log any optional permissions that were denied (for debugging)
                let allDeniedPermissions = getAllDeniedPermissions()
                let deniedOptionalPermissions = allDeniedPermissions.filter { !deniedEssentialPermissions.contains($0) }
                if !deniedOptionalPermissions.isEmpty {
                    print("Optional permissions were denied: \(deniedOptionalPermissions)")
                    print("App will continue with available data")
                }
                
                throw HealthKitError.partialDataAvailable(missingPermissions)
            }
        } catch {
            print("HealthKit authorization failed: \(error.localizedDescription)")
            lastError = error as? HealthKitError ?? HealthKitError.authorizationFailed
            throw error
        }
        
        return nil
    }
    
    deinit {
    }
    
    /// Get a list of missing permissions for better user feedback
    func getMissingPermissions() -> [String] {
        let allTypes: [(HKObjectType, String)] = [
            (HKQuantityType.quantityType(forIdentifier: .restingHeartRate)!, "Resting Heart Rate"),
            (HKQuantityType.quantityType(forIdentifier: .heartRateVariabilitySDNN)!, "Heart Rate Variability"),
            (HKQuantityType.quantityType(forIdentifier: .vo2Max)!, "VO2 Max"),
            (HKQuantityType.quantityType(forIdentifier: .bodyMass)!, "Body Mass"),
            (HKQuantityType.quantityType(forIdentifier: .bodyMassIndex)!, "Body Mass Index"),
            (HKQuantityType.quantityType(forIdentifier: .stepCount)!, "Step Count"),
            (HKCategoryType.categoryType(forIdentifier: .sleepAnalysis)!, "Sleep Analysis"),
            (HKCharacteristicType.characteristicType(forIdentifier: .biologicalSex)!, "Biological Sex"),
            (HKCharacteristicType.characteristicType(forIdentifier: .dateOfBirth)!, "Date of Birth")
        ]
        
        return allTypes.compactMap { (type, name) in
            let status = healthStore.authorizationStatus(for: type)
            return status != .sharingAuthorized ? name : nil
        }
    }
    
    /// Get a list of denied permissions (user previously denied them)
    /// Only checks essential permissions to avoid false warnings for optional permissions
    func getDeniedPermissions() -> [String] {
        let essentialTypes: [(HKObjectType, String)] = [
            (HKQuantityType.quantityType(forIdentifier: .restingHeartRate)!, "Resting Heart Rate"),
            (HKQuantityType.quantityType(forIdentifier: .stepCount)!, "Step Count"),
            (HKCharacteristicType.characteristicType(forIdentifier: .biologicalSex)!, "Biological Sex"),
            (HKCharacteristicType.characteristicType(forIdentifier: .dateOfBirth)!, "Date of Birth")
        ]
        
        return essentialTypes.compactMap { (type, name) in
            let status = healthStore.authorizationStatus(for: type)
            return status == .sharingDenied ? name : nil
        }
    }
    
    /// Get a list of all denied permissions (including optional ones)
    func getAllDeniedPermissions() -> [String] {
        let allTypes: [(HKObjectType, String)] = [
            (HKQuantityType.quantityType(forIdentifier: .restingHeartRate)!, "Resting Heart Rate"),
            (HKQuantityType.quantityType(forIdentifier: .heartRateVariabilitySDNN)!, "Heart Rate Variability"),
            (HKQuantityType.quantityType(forIdentifier: .vo2Max)!, "VO2 Max"),
            (HKQuantityType.quantityType(forIdentifier: .bodyMass)!, "Body Mass"),
            (HKQuantityType.quantityType(forIdentifier: .bodyMassIndex)!, "Body Mass Index"),
            (HKQuantityType.quantityType(forIdentifier: .stepCount)!, "Step Count"),
            (HKCategoryType.categoryType(forIdentifier: .sleepAnalysis)!, "Sleep Analysis"),
            (HKCharacteristicType.characteristicType(forIdentifier: .biologicalSex)!, "Biological Sex"),
            (HKCharacteristicType.characteristicType(forIdentifier: .dateOfBirth)!, "Date of Birth")
        ]
        
        return allTypes.compactMap { (type, name) in
            let status = healthStore.authorizationStatus(for: type)
            return status == .sharingDenied ? name : nil
        }
    }
    
    /// Debug method to print current permission status for all requested types
    func printPermissionStatus() {
        let allTypes: [(HKObjectType, String)] = [
            (HKQuantityType.quantityType(forIdentifier: .restingHeartRate)!, "Resting Heart Rate"),
            (HKQuantityType.quantityType(forIdentifier: .heartRateVariabilitySDNN)!, "Heart Rate Variability"),
            (HKQuantityType.quantityType(forIdentifier: .vo2Max)!, "VO2 Max"),
            (HKQuantityType.quantityType(forIdentifier: .bodyMass)!, "Body Mass"),
            (HKQuantityType.quantityType(forIdentifier: .bodyMassIndex)!, "Body Mass Index"),
            (HKQuantityType.quantityType(forIdentifier: .stepCount)!, "Step Count"),
            (HKCategoryType.categoryType(forIdentifier: .sleepAnalysis)!, "Sleep Analysis"),
            (HKCharacteristicType.characteristicType(forIdentifier: .biologicalSex)!, "Biological Sex"),
            (HKCharacteristicType.characteristicType(forIdentifier: .dateOfBirth)!, "Date of Birth")
        ]
        
        print("=== HealthKit Permission Status ===")
        for (type, name) in allTypes {
            let status = healthStore.authorizationStatus(for: type)
            let statusDescription = getAuthorizationStatusDescription(status)
            let isEssential = ["Resting Heart Rate", "Step Count", "Biological Sex", "Date of Birth"].contains(name)
            let essentialMarker = isEssential ? " [ESSENTIAL]" : " [OPTIONAL]"
            print("\(name)\(essentialMarker): \(statusDescription)")
        }
        print("===================================")
    }
    
    /// Check if permissions have been recovered (user manually enabled them in Health app)
    func checkPermissionRecovery() -> Bool {
        let deniedEssentialPermissions = getDeniedPermissions()
        if deniedEssentialPermissions.isEmpty {
            print("All essential permissions are now available!")
            return true
        } else {
            print("Essential permissions still denied: \(deniedEssentialPermissions)")
            return false
        }
    }
    
    /// Attempt to recover from denied permissions by checking current status
    @MainActor
    func attemptPermissionRecovery() async throws -> HealthDataSnapshot? {
        print("Attempting permission recovery...")
        
        // Check if permissions have been recovered
        if checkPermissionRecovery() {
            print("Permissions recovered! Fetching health data...")
            let healthData = try await fetchLatestHealthData()
            isAuthorized = true
            return healthData
        } else {
            let deniedEssentialPermissions = getDeniedPermissions()
            throw HealthKitError.permissionsDenied(deniedEssentialPermissions)
        }
    }
    
    /// Open Health app settings (if possible on watchOS)
    func openHealthAppSettings() {
        // On watchOS, we can't directly open the Health app settings
        // The user will need to manually navigate to Settings > Privacy & Security > Health
        print("Cannot programmatically open Health app on watchOS")
        print("User must manually navigate to: Settings > Privacy & Security > Health > Apps & Services > LifeClock")
    }
    
    /// Create a test health data snapshot with sample data for development/testing
    func createTestHealthDataSnapshot() -> HealthDataSnapshot {
        print("Creating test health data snapshot for development")
        
        return HealthDataSnapshot(
            restingHeartRate: 65.0,
            heartRateVariabilitySDNN: 45.0,
            vo2Max: 42.5,
            bodyMass: 70.0,
            bodyMassIndex: 22.5,
            stepCount: 8500.0,
            sleepAnalysis: SleepData(
                totalSleepHours: 7.5,
                deepSleepPercentage: 20.0,
                remSleepPercentage: 25.0,
                sleepEfficiency: 0.85
            ),
            biologicalSex: .male,
            dateOfBirth: Calendar.current.date(byAdding: .year, value: -30, to: Date()),
            dataSource: "Test Data"
        )
    }
    
    /// Check if we can use cached health data instead of re-running authorization
    func canUseCachedTestData() -> Bool {
        // If we have current health data and permissions are granted
        if let currentData = currentHealthData,
           hasRequiredPermissions() {
            return true
        }
        return false
    }
    
    // MARK: - Main Data Fetching Method
    @MainActor
    func fetchLatestHealthData() async throws -> HealthDataSnapshot {
        isLoading = true
        defer { isLoading = false }
        
        guard isHealthKitAvailable else {
            throw HealthKitError.notAvailable
        }
        
        do {
            // Fetch all health metrics concurrently
            async let restingHeartRate = fetchRestingHeartRate()
            async let heartRateVariability = fetchHeartRateVariability()
            async let vo2Max = fetchVO2Max()
            async let bodyMass = fetchBodyMass()
            async let bodyMassIndex = fetchBodyMassIndex()
            async let stepCount = fetchStepCount()
            async let sleepData = fetchSleepAnalysis()
            async let biologicalSex = fetchBiologicalSex()
            async let dateOfBirth = fetchDateOfBirth()
            
            // Wait for all results
            let healthData = HealthDataSnapshot(
                restingHeartRate: try await restingHeartRate,
                heartRateVariabilitySDNN: try await heartRateVariability,
                vo2Max: try await vo2Max,
                bodyMass: try await bodyMass,
                bodyMassIndex: try await bodyMassIndex,
                stepCount: try await stepCount,
                sleepAnalysis: try await sleepData,
                biologicalSex: try await biologicalSex,
                dateOfBirth: try await dateOfBirth
            )
            
            currentHealthData = healthData
            lastUpdateTime = Date()
            lastError = nil
            
            // Handle partial data scenarios
            if !healthData.missingDataTypes.isEmpty {
                let missingTypes = Array(healthData.missingDataTypes)
                print("Partial data available. Missing: \(missingTypes.joined(separator: ", "))")
                
                // If we're missing critical data and this is likely a simulator/test environment,
                // or if essential permissions were denied, provide test data for development purposes
                let deniedEssentialPermissions = getDeniedPermissions()
                if healthData.missingDataTypes.count >= 5 || !deniedEssentialPermissions.isEmpty {
                    if !deniedEssentialPermissions.isEmpty {
                        print("Essential permissions were denied: \(deniedEssentialPermissions). Using test data for development.")
                    } else {
                        print("Most data missing - likely simulator environment. Using test data for development.")
                    }
                    let testData = createTestHealthDataSnapshot()
                    currentHealthData = testData
                    lastUpdateTime = Date()
                    lastError = nil
                    return testData
                }
            } else {
                print("All health data successfully fetched!")
            }
            
            print("Health data snapshot: \(healthData)")
            
            return healthData
            
        } catch {
            let healthKitError = error as? HealthKitError ?? HealthKitError.dataFetchFailed
            lastError = healthKitError
            throw healthKitError
        }
    }
    
    // MARK: - Individual Data Fetching Methods
    private func fetchRestingHeartRate() async throws -> Double? {
        guard let type = HKQuantityType.quantityType(forIdentifier: .restingHeartRate) else { return nil }
        return try await fetchLatestQuantitySample(for: type, unit: HKUnit(from: "count/min"))
    }
    
    private func fetchHeartRateVariability() async throws -> Double? {
        guard let type = HKQuantityType.quantityType(forIdentifier: .heartRateVariabilitySDNN) else { return nil }
        return try await fetchLatestQuantitySample(for: type, unit: HKUnit.secondUnit(with: .milli))
    }
    
    private func fetchVO2Max() async throws -> Double? {
        guard let type = HKQuantityType.quantityType(forIdentifier: .vo2Max) else { return nil }
        return try await fetchLatestQuantitySample(for: type, unit: HKUnit(from: "ml/kg*min"))
    }
    
    private func fetchBodyMass() async throws -> Double? {
        guard let type = HKQuantityType.quantityType(forIdentifier: .bodyMass) else { return nil }
        return try await fetchLatestQuantitySample(for: type, unit: HKUnit.gramUnit(with: .kilo))
    }
    
    private func fetchBodyMassIndex() async throws -> Double? {
        guard let type = HKQuantityType.quantityType(forIdentifier: .bodyMassIndex) else { return nil }
        return try await fetchLatestQuantitySample(for: type, unit: HKUnit.count())
    }
    
    private func fetchStepCount() async throws -> Double? {
        guard let type = HKQuantityType.quantityType(forIdentifier: .stepCount) else { return nil }
        // Get today's step count
        let calendar = Calendar.current
        let startOfDay = calendar.startOfDay(for: Date())
        let endOfDay = calendar.date(byAdding: .day, value: 1, to: startOfDay) ?? Date()
        
        return try await fetchSummedQuantitySample(for: type, 
                                                 unit: HKUnit.count(), 
                                                 startDate: startOfDay, 
                                                 endDate: endOfDay)
    }
    
    private func fetchSleepAnalysis() async throws -> SleepData? {
        guard let sleepType = HKCategoryType.categoryType(forIdentifier: .sleepAnalysis) else { 
            print("Sleep analysis type not available")
            return nil 
        }
        
        let calendar = Calendar.current
        let endDate = Date()
        let startDate = calendar.date(byAdding: .day, value: -1, to: endDate) ?? endDate
        
        let predicate = HKQuery.predicateForSamples(withStart: startDate, end: endDate, options: .strictStartDate)
        
        return try await withCheckedThrowingContinuation { continuation in
            let query = HKSampleQuery(sampleType: sleepType, predicate: predicate, limit: HKObjectQueryNoLimit, sortDescriptors: nil) { _, samples, error in
                if let error = error {
                    print("Error fetching sleep analysis: \(error.localizedDescription)")
                    continuation.resume(throwing: error)
                    return
                }
                
                guard let sleepSamples = samples as? [HKCategorySample] else {
                    print("No sleep analysis data found")
                    continuation.resume(returning: nil)
                    return
                }
                
                print("Found \(sleepSamples.count) sleep samples")
                
                var totalSleepTime: TimeInterval = 0
                
                for sample in sleepSamples {
                    if sample.value == HKCategoryValueSleepAnalysis.asleepUnspecified.rawValue {
                        totalSleepTime += sample.endDate.timeIntervalSince(sample.startDate)
                    }
                }
                
                let sleepHours = totalSleepTime / 3600.0
                print("Total sleep time: \(sleepHours) hours")
                
                let sleepData = SleepData(
                    totalSleepHours: sleepHours,
                    deepSleepPercentage: nil,
                    remSleepPercentage: nil,
                    sleepEfficiency: sleepHours > 0 ? min(1.0, sleepHours / 8.0) : nil
                )
                
                continuation.resume(returning: sleepData)
            }
            
            self.healthStore.execute(query)
        }
    }
    
    private func fetchBiologicalSex() async throws -> HKBiologicalSex? {
        do {
            let biologicalSex = try healthStore.biologicalSex()
            print("Biological sex fetched: \(biologicalSex.biologicalSex)")
            return biologicalSex.biologicalSex
        } catch {
            print("Failed to fetch biological sex: \(error.localizedDescription)")
            return nil
        }
    }
    
    private func fetchDateOfBirth() async throws -> Date? {
        do {
            let dateOfBirth = try healthStore.dateOfBirthComponents()
            let date = Calendar.current.date(from: dateOfBirth)
            print("Date of birth fetched: \(date?.description ?? "nil")")
            return date
        } catch {
            print("Failed to fetch date of birth: \(error.localizedDescription)")
            return nil
        }
    }
    
    // MARK: - Helper Methods
    private func fetchLatestQuantitySample(for quantityType: HKQuantityType, unit: HKUnit) async throws -> Double? {
        let predicate = HKQuery.predicateForSamples(
            withStart: Calendar.current.date(byAdding: .month, value: -1, to: Date()),
            end: Date(),
            options: .strictEndDate
        )
        
        return try await withCheckedThrowingContinuation { continuation in
            let query = HKSampleQuery(
                sampleType: quantityType,
                predicate: predicate,
                limit: 1,
                sortDescriptors: [NSSortDescriptor(key: HKSampleSortIdentifierStartDate, ascending: false)]
            ) { _, samples, error in
                if let error = error {
                    print("Error fetching \(quantityType.identifier): \(error.localizedDescription)")
                    continuation.resume(throwing: error)
                    return
                }
                
                guard let sample = samples?.first as? HKQuantitySample else {
                    print("No data found for \(quantityType.identifier)")
                    continuation.resume(returning: nil)
                    return
                }
                
                let value = sample.quantity.doubleValue(for: unit)
                print("Fetched \(quantityType.identifier): \(value)")
                continuation.resume(returning: value)
            }
            
            self.healthStore.execute(query)
        }
    }
    
    private func fetchSummedQuantitySample(for quantityType: HKQuantityType, unit: HKUnit, startDate: Date, endDate: Date) async throws -> Double? {
        let predicate = HKQuery.predicateForSamples(withStart: startDate, end: endDate, options: .strictStartDate)
        
        return try await withCheckedThrowingContinuation { continuation in
            let query = HKStatisticsQuery(
                quantityType: quantityType,
                quantitySamplePredicate: predicate,
                options: .cumulativeSum
            ) { _, result, error in
                if let error = error {
                    print("Error fetching summed \(quantityType.identifier): \(error.localizedDescription)")
                    continuation.resume(throwing: error)
                    return
                }
                
                guard let result = result, let sum = result.sumQuantity() else {
                    print("No summed data found for \(quantityType.identifier)")
                    continuation.resume(returning: nil)
                    return
                }
                
                let value = sum.doubleValue(for: unit)
                print("Fetched summed \(quantityType.identifier): \(value)")
                continuation.resume(returning: value)
            }
            
            self.healthStore.execute(query)
        }
    }
    
    
    
    
}

enum HealthKitError: Error, LocalizedError {
    case notAvailable
    case authorizationFailed
    case dataFetchFailed
    case permissionDenied(String)
    case permissionsDenied([String])
    case dataUnavailable(String)
    case partialDataAvailable([String])
    
    var errorDescription: String? {
        switch self {
        case .notAvailable:
            return "HealthKit is not available on this device"
        case .authorizationFailed:
            return "Failed to authorize HealthKit access. Please check your privacy settings."
        case .dataFetchFailed:
            return "Unable to fetch health data. Please try again."
        case .permissionDenied(let dataType):
            return "Permission denied for \(dataType). Please enable access in Health app settings."
        case .permissionsDenied(let deniedTypes):
            return "Health permissions were previously denied. Please go to Health app → Privacy → Apps & Services → LifeClock and enable all permissions to use this app."
        case .dataUnavailable(let dataType):
            return "No data available for \(dataType). Please ensure data is recorded in the Health app."
        case .partialDataAvailable(let missingTypes):
            return "Some health data is unavailable: \(missingTypes.joined(separator: ", ")). Life expectancy calculations may be less accurate."
        }
    }
}